<?php
/**
 * Class that add wish list functionality
 *
 * @package Codeinwp\Sparks\Modules\Wish_List
 */

namespace Codeinwp\Sparks\Modules\Wish_List;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Codeinwp\Sparks\Core\Dynamic_Styles;
use Codeinwp\Sparks\Modules\Base_Module;
use Codeinwp\Sparks\Core\Traits\Common as Trait_Common;
use Codeinwp\Sparks\Core\Traits\Conditional_Asset_Loading_Utilities;

/**
 * Class Wish_List
 */
class Wish_List extends Base_Module {
	use Trait_Common, Conditional_Asset_Loading_Utilities;

	const SETTING_BUTTON_POSITION = 'btn_position';

	/**
	 * Default module activation status
	 *
	 * @var bool
	 */
	protected $default_status = true;

	/**
	 * If module has configuration options or not.
	 *
	 * @var bool
	 */
	protected $has_dashboard_config = true;

	/**
	 * Define module setting prefix.
	 *
	 * @var string
	 */
	protected $setting_prefix = 'wl';

	/**
	 * Define module slug.
	 *
	 * @var string
	 */
	protected $module_slug = 'wish_list';

	/**
	 * Cookie id.
	 *
	 * @var string
	 */
	public $cookie_id = 'nv-wishlist';

	/**
	 * Help URL
	 *
	 * @var string
	 */
	protected $help_url = '';

	/**
	 * Get Module Name
	 *
	 * @return string
	 */
	public function get_name() {
		return esc_html__( 'Product Wishlist', 'sparks-for-woocommerce' );
	}

	/**
	 * Should load the module.
	 *
	 * @return bool
	 */
	public function should_load() {
		return $this->get_status();
	}

	/**
	 * Get dashboard description
	 *
	 * @return string
	 */
	public function get_dashboard_description() {
		return esc_html__( 'Loyalize your customers by saving their favourite products, finding them quickly and easily at a later time to buy them.', 'sparks-for-woocommerce' );
	}

	/**
	 * Register wish list hooks.
	 *
	 * @return mixed|void
	 */
	public function init() {
		// If the action is set to 'init' instead of 'wp', If you disable the wishlist, save in customizer then try to add it back, the preview won't render it first time.
		add_action( 'wp', array( $this, 'run' ) );
		if ( $this->get_button_position() === 'none' ) {
			return null;
		}
		add_action( 'rest_api_init', array( $this, 'register_endpoints' ) );
		add_filter( 'rest_request_before_callbacks', array( $this, 'load_cart_before_wishlist_rendering' ), 10, 2 );

		add_action( 'woocommerce_account_menu_items', array( $this, 'add_account_tab' ) );
		add_action( 'init', array( $this, 'add_wish_list_endpoint' ) );
		add_filter( 'query_vars', array( $this, 'wish_list_query_vars' ), 0 );
		add_action( 'woocommerce_account_sp-wish-list_endpoint', array( $this, 'render_wish_list_table' ) );
		add_action( 'wp_login', array( $this, 'update_wishlist_from_cookie' ), 10, 2 );
		add_action( 'woocommerce_after_add_to_cart_button', array( $this, 'add_wish_list_button' ) );
		add_filter( 'neve_selectors_buttons_secondary_normal', array( $this, 'add_secondary_btns_normal' ) );
		add_filter( 'neve_selectors_buttons_secondary_hover', array( $this, 'add_secondary_btns_hover' ) );
		add_filter( 'neve_selectors_buttons_secondary_padding', array( $this, 'add_secondary_btns_padding' ) );
		add_action( 'wp_enqueue_scripts', array( $this, 'register_assets' ) );
		add_action( 'sparks_wish_list_icon', array( $this, 'render_wish_list_icon' ) );
	}

	/**
	 * Load WC Cart before Wishlist rendering.
	 *
	 * @param  WP_REST_Response|WP_HTTP_Response|WP_Error|mixed $response That current response of the callback.
	 * @return WP_REST_Response|WP_HTTP_Response|WP_Error|mixed
	 */
	public function load_cart_before_wishlist_rendering( $response, $handler ) {
		// run only for get_products callback.
		if ( is_array( $handler['callback'] ) && array( $this, 'get_product' ) === $handler['callback'] ) {
			// For Only Quick View: load WC cart to initialize the Session. (Normally, it's initialized by WC for is_request is frontend. This REST API call requires manual load Session class. )
			wc_load_cart();
		}

		return $response;
	}

	/**
	 * Register endpoints.
	 */
	public function register_endpoints() {
		/**
		 * Wish List update endpoint.
		 */
		register_rest_route(
			SPARKS_WC_REST_NAMESPACE,
			'/update_wishlist/',
			array(
				'methods'             => \WP_REST_Server::CREATABLE,
				'callback'            => array( $this, 'update_wishlist' ),
				'permission_callback' => function () {
					return is_user_logged_in();
				},
			)
		);
	}

	/**
	 * Update the wishlist.
	 *
	 * @param \WP_REST_Request $request the rest request.
	 *
	 * @return \WP_REST_Response
	 */
	public function update_wishlist( \WP_REST_Request $request ) {
		$user_id       = get_current_user_id();
		$data          = $request->get_json_params() ? $request->get_json_params() : array();
		$current_value = $this->get_meta_wishlist_array( $user_id );

		if ( is_array( $current_value ) && ! empty( $current_value ) ) {
			$data = array_replace( $current_value, $data );
		}

		$data = array_filter( $data );

		if ( count( $data ) >= 50 ) {
			$first_element = array_keys( $data );
			unset( $data[ $first_element[0] ] );
		}

		update_user_meta( $user_id, 'wish_list_products', wp_json_encode( $data ) );

		return new \WP_REST_Response(
			array(
				'code'    => 'success',
				'message' => esc_html__( 'Wishlist updated', 'sparks-for-woocommerce' ),
				'data'    => $data,
			)
		);
	}

	/**
	 * Register Dynamic Styles
	 *
	 * @return void
	 */
	public function register_dynamic_styles() {
		$default_colors = sparks_current_theme()->wish_list()->default_colors();

		Dynamic_Styles::get_instance()->push(
			'.products .sp-wl-wrap .add-to-wl',
			[
				'background' => $default_colors->get( 'catalog_add_btn_bg' ),
			]
		);

		Dynamic_Styles::get_instance()->push(
			'.products .sp-wl-wrap .add-to-wl.item-added',
			[
				'background' => $default_colors->get( 'catalog_add_btn_added_bg' ),
			]
		);

		Dynamic_Styles::get_instance()->push(
			'.sp-wl-product-wrap .add-to-wl',
			[
				'border' => sprintf( '2px solid %s', $default_colors->get( 'single_add_btn_border' ) ),
				'color'  => $default_colors->get( 'single_add_btn_text' ),
			]
		);

		Dynamic_Styles::get_instance()->push(
			'.sp-wl-product-wrap .add-to-wl:hover',
			[
				'background' => $default_colors->get( 'single_add_btn_hover_bg' ),
			]
		);

		Dynamic_Styles::get_instance()->push(
			'.sp-wl-notification',
			[
				'background' => $default_colors->get( 'notification_bg' ),
			]
		);

		Dynamic_Styles::get_instance()->push(
			'.sp-wl-product',
			[
				'border-bottom' => sprintf( '3px solid %s', $default_colors->get( 'my_account_row_bottom_border' ) ),
			]
		);

		Dynamic_Styles::get_instance()->push(
			'.sp-wl-notification path',
			[
				'fill' => $default_colors->get( 'notification_icon_bg' ),
			]
		);
	}

	/**
	 * Should the assets be loaded?
	 *
	 * @return bool
	 */
	protected function needs_frontend_assets() {
		return $this->current_page_has_loop_products() || is_account_page(); // note for is_account_page() condition; we can restrict the statement with only wishlist tab of the my account page, in the future.
	}

	/**
	 * Register styles and script assets
	 */
	final public function register_assets() {
		if ( ! $this->needs_frontend_assets() ) {
			return false;
		}

		sparks_enqueue_style( 'sparks-wl-style', SPARKS_WC_URL . 'includes/assets/wish_list/css/style.min.css', array(), SPARKS_WC_VERSION );
		sparks_enqueue_script( 'sparks-wl-script', SPARKS_WC_URL . 'includes/assets/wish_list/js/build/app.js', array(), SPARKS_WC_VERSION, true );

		$url = wc_get_endpoint_url( 'sp-wish-list', '', get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) );

		wp_localize_script(
			'sparks-wl-script',
			'sparkWl',
			[
				'loggedIn'       => is_user_logged_in(),
				'updateEndpoint' => rest_url( SPARKS_WC_REST_NAMESPACE . '/update_wishlist/' ),
				'nonce'          => wp_create_nonce( 'wp_rest' ),
				'i18n'           => [
					'error'            => esc_html__( 'There was an error while trying to update the wishlist.', 'sparks-for-woocommerce' ),
					'empty'            => esc_html__( 'You don\'t have any products in your wish list yet.', 'sparks-for-woocommerce' ),
					/* translators: %s - url */
					'noticeTextAdd'    => sprintf( esc_html__( 'This product has been added to your %s.', 'sparks-for-woocommerce' ), sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url ), esc_html__( 'wish list', 'sparks-for-woocommerce' ) ) ),
					/* translators: %s - url */
					'noticeTextRemove' => sprintf( esc_html__( 'This product has been removed from your %s.', 'sparks-for-woocommerce' ), sprintf( '<a href="%1$s">%2$s</a>', esc_url( $url ), esc_html__( 'wish list', 'sparks-for-woocommerce' ) ) ),
				],
			]
		);
	}

	/**
	 * Get wishlist button position
	 *
	 * @return string
	 */
	public function get_button_position() {
		return $this->get_setting( self::SETTING_BUTTON_POSITION, 'none' );
	}

	/**
	 * Updates wish list from $_COOKIE.
	 *
	 * @param  string   $user_login the user name.
	 * @param \WP_User $user       user object.
	 */
	public function update_wishlist_from_cookie( $user_login, \WP_User $user ) {
		$meta_wish_list = $this->get_meta_wishlist_array( $user->ID );

		if ( empty( $meta_wish_list ) ) {
			$meta_wish_list = array();
		}

		if ( ! isset( $_COOKIE[ $this->cookie_id ] ) || ! is_array( $this->get_cookie_wishlist_array() ) ) {
			return;
		}

		$meta_wish_list = array_replace( $meta_wish_list, $this->get_cookie_wishlist_array() );

		if ( count( $meta_wish_list ) >= 50 ) {
			$first_element = array_keys( $meta_wish_list );
			unset( $meta_wish_list[ $first_element[0] ] );
		}

		update_user_meta( $user->ID, 'wish_list_products', wp_json_encode( $meta_wish_list ) );
		setcookie( $this->cookie_id, '', - 1, '/' ); //phpcs:ignore WordPressVIPMinimum.Functions.RestrictedFunctions.cookies_setcookie
	}

	/**
	 * Run wish list actions.
	 */
	public function run() {
		if ( $this->get_button_position() === 'none' ) {
			return false;
		}
		add_action( 'sparks_product_actions', array( $this, 'add_wish_list_button' ) );
		add_action( 'wp_footer', array( $this, 'render_wl_notifications' ) );
	}

	/**
	 * Checks if the product is in the wishlist.
	 *
	 * @param int $product_id the product id.
	 *
	 * @return bool
	 */
	private function is_product_in_wishlist( $product_id ) {
		$user_id          = get_current_user_id();
		$cookie_wish_list = $this->get_cookie_wishlist_array();
		if ( 0 !== $user_id ) {
			$wish_list = $this->get_meta_wishlist_array( $user_id );
			$wish_list = array_replace( $wish_list, $cookie_wish_list );

			if ( ! empty( $wish_list ) && isset( $wish_list[ $product_id ] ) ) {
				return $wish_list[ $product_id ];
			}

			return false;
		}

		if ( array_key_exists( $product_id, $cookie_wish_list ) ) {
			return $cookie_wish_list[ $product_id ];
		}

		return false;
	}

	/**
	 * Get wish list from cookie.
	 *
	 * @return array
	 */
	private function get_cookie_wishlist_array() {
		if ( ! isset( $_COOKIE[ $this->cookie_id ] ) ) {
			return array();
		}

		$cookie_wishlist = json_decode( wp_unslash( sanitize_text_field( $_COOKIE[ $this->cookie_id ] ) ), true ); //phpcs:ignore WordPressVIPMinimum.Variables.RestrictedVariables.cache_constraints___COOKIE

		if ( ! is_array( $cookie_wishlist ) ) {
			return array();
		}

		return $cookie_wishlist;
	}

	/**
	 * Get wish list from user meta.
	 *
	 * @return array
	 */
	public function get_meta_wishlist_array( $user_id ) {
		$meta_wishlist = json_decode( get_user_meta( $user_id, 'wish_list_products', true ), true );
		if ( ! is_array( $meta_wishlist ) ) {
			return array();
		}

		return $meta_wishlist;
	}

	/**
	 * Wish List button markup.
	 */
	public function add_wish_list_button() {
		if ( sparks_is_amp() ) {
			return;
		}
		global $product;

		$position   = $this->get_button_position();
		$product_id = $product->get_id();

		// deprecated since v1.0.0 and will be removed with v1.4.0 permanently, please use "sparks_wish_list_label"
		$wish_list_label = apply_filters( 'neve_wish_list_label', __( 'Add to wishlist', 'sparks-for-woocommerce' ) );

		// throw notice about deprecated WP filter.
		sparks_notice_deprecated_filter( 'neve_wish_list_label', 'sparks_wish_list_label', '1.0.0' );

		$wish_list_label = apply_filters( 'sparks_wish_list_label', $wish_list_label );

		// deprecated since v1.0.0 and will be removed with v1.4.0 permanently, please use "sparks_sr_title"
		$title_sr = apply_filters(
			'neve_sr_title',
			/* translators: %s - product title */
			sprintf( __( 'Add %s to wishlist', 'sparks-for-woocommerce' ), get_the_title() )
		);

		// throw notice about deprecated WP filter.
		sparks_notice_deprecated_filter( 'neve_sr_title', 'sparks_sr_title', '1.0.0' );

		$title_sr = apply_filters( 'sparks_sr_title', $title_sr );

		$icon_class = array( 'add-to-wl' );
		if ( $this->is_product_in_wishlist( $product_id ) ) {
			$icon_class[] = 'item-added';
		}

		$wl_wrapper_classes = array( 'sp-wl-wrap' );
		$style              = 'top' === $position ? 'order: 1; align-self: start;' : 'order: 2; align-self: end;';
		if ( 'woocommerce_after_add_to_cart_button' === current_filter() ) {
			$wl_wrapper_classes[] = 'sp-wl-product-wrap';
			$style                = '';
		}

		echo '<div class="' . esc_attr( implode( ' ', $wl_wrapper_classes ) ) . '" ';
		if ( ! empty( $style ) ) {
			echo 'style="' . esc_attr( $style ) . '"';
		}
		echo '>';

		echo '<div class="' . esc_attr( implode( ' ', $icon_class ) ) . '" data-pid="' . esc_attr( $product_id ) . '" aria-label="' . esc_attr( $title_sr ) . '" tabindex="0">';
		echo '<svg width="16" height="16" viewBox="0 0 512 512"><path xmlns="http://www.w3.org/2000/svg" fill="currentColor" d="M462.3 62.6C407.5 15.9 326 24.3 275.7 76.2L256 96.5l-19.7-20.3C186.1 24.3 104.5 15.9 49.7 62.6c-62.8 53.6-66.1 149.8-9.9 207.9l193.5 199.8c12.5 12.9 32.8 12.9 45.3 0l193.5-199.8c56.3-58.1 53-154.3-9.8-207.9z"/></svg>';
		if ( 'woocommerce_after_add_to_cart_button' !== current_filter() ) {
			echo '<span class="tooltip tooltip-left">' . esc_html( $wish_list_label ) . '</span>';
		}
		echo '</div>';
		echo '</div>';
	}

	/**
	 * Wish list icon markup.
	 *
	 * @param array $settings Settings array.
	 *
	 * @return void
	 */
	public function render_wish_list_icon( $settings = array() ) {
		$params = [];

		$params['tag']   = ! empty( $settings['tag'] ) ? $settings['tag'] : 'li';
		$params['class'] = ! empty( $settings['class'] ) ? $settings['class'] : 'menu-item-nav-wish-list';
		$params['label'] = ! empty( $settings['label'] ) ? $settings['label'] : '';

		$params['url'] = wc_get_endpoint_url( 'sp-wish-list', '', get_permalink( get_option( 'woocommerce_myaccount_page_id' ) ) );

		sparks_get_template( 'wish_list', 'icon', $params );
	}

	/**
	 * Register new endpoint to use for My Account page
	 */
	public function add_wish_list_endpoint() {
		add_rewrite_endpoint( 'sp-wish-list', EP_ROOT | EP_PAGES );
		$this->maybe_flush_rules( 'wishlist' );
	}

	/**
	 * Add new query var
	 *
	 * @param array $vars Query vars.
	 *
	 * @return array
	 */
	public function wish_list_query_vars( $vars ) {
		$vars[] = 'sp-wish-list';

		return $vars;
	}

	/**
	 * Add Wish List tab in account page.
	 *
	 * @param array $items WooCommerce tabs.
	 *
	 * @return array
	 */
	public function add_account_tab( $items ) {
		$logout = $items['customer-logout'];
		unset( $items['customer-logout'] );
		$items['sp-wish-list']    = esc_html__( 'Wish List', 'sparks-for-woocommerce' );
		$items['customer-logout'] = $logout;

		return $items;
	}

	/**
	 * Render wish list in My account page.
	 */
	public function render_wish_list_table() {
		$user_id            = get_current_user_id();
		$wish_list_products = array_filter( array_replace( $this->get_meta_wishlist_array( $user_id ), $this->get_cookie_wishlist_array() ) );
		if ( empty( $wish_list_products ) ) {
			// deprecated since v1.0.0 and will be removed with v1.4.0 permanently, please use "sparks_wishlist_empty"
			$empty_text = apply_filters( 'neve_wishlist_empty', __( 'You don\'t have any products in your wish list yet.', 'sparks-for-woocommerce' ) );

			// throw notice about deprecated WP filter.
			sparks_notice_deprecated_filter( 'neve_wishlist_empty', 'sparks_wishlist_empty', '1.0.0' );

			echo esc_html( apply_filters( 'sparks_wishlist_empty', $empty_text ) );

			return;
		}

		echo '<div class="sp-wishlist-wrap">';
		foreach ( $wish_list_products as $pid => $enabled ) {
			$product = wc_get_product( $pid );
			if ( ! ( $product instanceof \WC_Product ) ) {
				continue;
			}
			$availability = $product->get_availability();
			$stock_status = isset( $availability['class'] ) ? $availability['class'] : false;

			echo '<div class="sp-wl-product">';
			echo '<div class="loader-wrap"><span class="sp-loader"></span></div>';
			echo '<div class="sp-wl-product-content">';

			echo '<div class="product-thumbnail">';
			echo '<a href="' . esc_url( get_permalink( apply_filters( 'woocommerce_in_cart_product', $pid ) ) ) . '">';
			echo wp_kses(
				$product->get_image( 'woocommerce_gallery_thumbnail' ),
				[
					'img' => [
						'width'    => [],
						'height'   => [],
						'src'      => [],
						'class'    => [],
						'alt'      => [],
						'decoding' => [],
						'loading'  => [],
						'srcset'   => [],
						'sizes'    => [],
					],
				]
			);
			echo '</a>';
			echo '</div>';

			echo '<div class="details">';

			echo '<div class="product-name">';
			echo '<a href="' . esc_url( get_permalink( apply_filters( 'woocommerce_in_cart_product', $pid ) ) ) . '">';
			echo esc_html( apply_filters( 'woocommerce_in_cartproduct_obj_title', $product->get_title(), $product ) );
			echo '</a>';
			echo '</div>';

			echo '<div class="price-stock">';
			echo '<div class="product-stock-status">';
			echo 'out-of-stock' === $stock_status ? '<span class="wishlist-out-of-stock">' . esc_html__( 'Out of Stock', 'sparks-for-woocommerce' ) . '</span>' : '<span class="wishlist-in-stock">' . esc_html__( 'In Stock', 'sparks-for-woocommerce' ) . '</span>';
			echo '</div>';

			echo '<div class="product-price">';
			// deprecated since v1.0.0 and will be removed with v1.4.0 permanently, please use "sparks_wishlist_table_free_text"
			$free_text = apply_filters( 'neve_wishlist_table_free_text', __( 'Free!', 'sparks-for-woocommerce' ), $product );

			// throw notice about deprecated WP filter.
			sparks_notice_deprecated_filter( 'neve_wishlist_table_free_text', 'sparks_wishlist_table_free_text', '1.0.0' );

			echo $product->get_price() ? wp_kses(
				$product->get_price_html(),
				[
					'del'  => [
						'aria-hidden' => [],
					],
					'span' => [
						'class' => [],
					],
					'bdi'  => [],
					'ins'  => [],
				]
			) : esc_html( apply_filters( 'sparks_wishlist_table_free_text', esc_html( $free_text ), $product ) );
			echo '</div>';
			echo '</div>'; // .price-stock

			echo '<div class="actions">';
			if ( ! empty( $stock_status ) && 'out-of-stock' !== $stock_status ) {
				echo '<div class="product-add-to-cart">';
				echo wp_kses(
					apply_filters(
						'woocommerce_loop_add_to_cart_link',
						sprintf(
							'<a href="%s" data-quantity="1" class="button button-primary">%s</a>',
							esc_url( $product->add_to_cart_url() ),
							esc_html( $product->add_to_cart_text() )
						),
						$product
					),
					[
						'a' => [
							'href'          => true,
							'data-quantity' => true,
							'class'         => true,
						],
					]
				);
				echo '</div>'; // .product-add-to-cart
			}

			echo '<a class="remove remove-wl-item" data-pid="' . esc_attr( $pid ) . '">';
			echo '<span class="dashicons dashicons-no-alt"></span>';
			echo '</a>';
			echo '</div>'; // .actions

			echo '</div>'; // .details
			echo '</div>'; // .sp-wl-product
			echo '</div>'; // .sp-wl-product-content
		}
		echo '</div>'; // .sp-wishlist-wrap
	}

	/**
	 * Render function for wish list notification
	 */
	public function render_wl_notifications() {
		if ( sparks_is_amp() || ! $this->needs_frontend_assets() ) {
			return;
		}
		echo '<div class="sp-wl-notification" role="dialog">';
		echo '<div class="wl-notification-icon">';
		echo '<svg width="50" height="50" viewBox="0 0 512 512"><path xmlns="http://www.w3.org/2000/svg" fill="currentColor" d="M462.3 62.6C407.5 15.9 326 24.3 275.7 76.2L256 96.5l-19.7-20.3C186.1 24.3 104.5 15.9 49.7 62.6c-62.8 53.6-66.1 149.8-9.9 207.9l193.5 199.8c12.5 12.9 32.8 12.9 45.3 0l193.5-199.8c56.3-58.1 53-154.3-9.8-207.9z"/></svg>';
		echo '</div>';
		echo '<div class="wl-notification-content">';
		echo '</div>';
		echo '</div>';
	}

	/**
	 * Add wishlist button to secondary buttons selectors.
	 *
	 * @param string $selector Secondary button selectors.
	 *
	 * @return string
	 */
	public function add_secondary_btns_normal( $selector ) {
		return ( $selector . ', .sp-wl-product-wrap .add-to-wl' );
	}

	/**
	 * Add wishlist button to secondary buttons hover selectors.
	 *
	 * @param string $selector Secondary button hover selectors.
	 *
	 * @return string
	 */
	public function add_secondary_btns_hover( $selector ) {
		return ( $selector . ', .sp-wl-product-wrap .add-to-wl:hover, .sp-wl-product-wrap .add-to-wl.item-added' );
	}

	/**
	 * Add wishlist button to secondary buttons padding selectors.
	 *
	 * @param string $selector Secondary button padding selectors.
	 *
	 * @return string
	 */
	public function add_secondary_btns_padding( $selector ) {
		return ( $selector . ', .sp-wl-product-wrap .add-to-wl' );
	}
}
